const database = require('./database');
const translator = require('./translator');
const commands = require('./commands');
const businessHours = require('./businessHours');
const notifications = require('./notifications');
const config = require('../config/config');
const mediaHandler = require('./mediaHandler');

class MessageHandler {
  constructor(client) {
    this.client = client;
  }

  /**
   * Get active agent phone numbers
   */
  getActiveAgents() {
    const agents = database.getAllAgents();
    return agents.map(agent => agent.phone);
  }

  /**
   * Set WPPConnect client
   */
  setClient(client) {
    this.client = client;
    notifications.setClient(client);
  }

  /**
   * Check if number is from an admin or agent
   */
  isAgent(phone) {
    const cleanPhone = phone.replace('@c.us', '');
    // Check if is admin
    if (config.admins.includes(cleanPhone)) {
      return true;
    }
    // Check if is registered agent
    const agent = database.getAgent(cleanPhone);
    return agent && agent.status === 'active';
  }

  /**
   * Process received message
   */
  async processMessage(message) {
    try {
      const phone = message.from.replace('@c.us', '');
      const isAgent = this.isAgent(phone);
      const content = message.body || '';
      const messageType = message.type;

      // Message log
      if (messageType === 'chat') {
        console.log(`📨 Message received from ${phone} (${isAgent ? 'Agent' : 'Client'}): ${content.substring(0, 50)}...`);
      } else {
        console.log(`📨 Media received from ${phone} (${isAgent ? 'Agent' : 'Client'}): [${messageType}]`);
      }

      // PRIORITY CHECK: If agent is in active flow (transfer, etc)
      // BUT allows commands to pass through (like /send, /cancel)
      if (isAgent && messageType === 'chat' && !commands.isCommand(content)) {
        const agentState = commands.getClientState(phone);
        
        // If agent is in transfer flow
        if (agentState && agentState.flow === 'transfer') {
          const result = await commands.processTransferFlow(phone, content);
          if (result) {
            await this.sendMessage(phone + '@c.us', result.response);
          }
          return;
        }
        
        // If agent is in transfer notification flow
        if (agentState && agentState.flow === 'transfer_notify') {
          const result = await commands.processTransferNotifyFlow(phone, content);
          if (result) {
            await this.sendMessage(phone + '@c.us', result.response);
            
            // If transfer was processed, notify
            if (result.action === 'transfer_pending') {
              await notifications.notifyAgentTransferPending(
                result.targetAgent,
                result.ticketId,
                result.fromAgent,
                result.clientPhone,
                result.clientName,
                result.reason
              );
              
              // If requested to notify client
              if (result.notifyClient) {
                const ticket = database.getTicket(result.ticketId);
                await notifications.notifyClientTransferred(
                  result.clientPhone,
                  result.targetAgent,
                  ticket.language
                );
              }
            }
          }
          return;
        }

        // If agent is in notification flow
        if (agentState && agentState.flow === 'notification') {
          const result = await commands.processNotificationFlow(phone, content);
          if (result) {
            await this.sendMessage(phone + '@c.us', result.response);
          }
          return;
        }
      }

      // If agent is in notification flow and sends media
      if (isAgent && messageType !== 'chat') {
        const agentState = commands.getClientState(phone);
        if (agentState && agentState.flow === 'notification') {
          const added = commands.addMediaToNotification(phone, {
            type: messageType,
            caption: message.caption || '',
            mimetype: message.mimetype,
            filename: message.filename
          });
          
          if (added) {
            await this.sendMessage(phone + '@c.us', `✅ Media added!\n\n💡 Type */send* to deliver notification.`);
            return;
          }
        }
      }

      // If it's media (not chat), process as media
      if (messageType !== 'chat') {
        // If it's agent, forward to client
        if (isAgent) {
          return await this.handleAgentMessage(phone, message);
        }
        // If it's client, process as client message (media in ticket)
        return await this.handleClientMessage(phone, message);
      }

      // If it's a command (only for text messages), process command
      if (commands.isCommand(content)) {
        return await this.handleCommand(phone, content, isAgent);
      }

      // Check if admin is in a flow (before treating as agent message)
      if (isAgent) {
        const adminCommands = require('./adminCommands');
        if (adminCommands.isAdmin(phone) && adminCommands.adminStates.has(phone)) {
          console.log(`🔄 Admin ${phone} is in a flow, processing as admin command`);
          const result = await adminCommands.processAdminFlow(phone, content);
          if (result && result.response) {
            await this.sendMessage(phone + '@c.us', result.response);
            return;
          }
        }
      }

      // If it's agent, forward to client
      if (isAgent) {
        return await this.handleAgentMessage(phone, message);
      }

      // If it's client, process client message
      return await this.handleClientMessage(phone, message);

    } catch (error) {
      console.error('❌ Error processing message:', error);
      await this.sendMessage(message.from, '❌ Error processing your message. Please try again.');
    }
  }

  /**
   * Process command
   */
  async handleCommand(phone, content, isAgent) {
    const result = await commands.processCommand(content, phone, isAgent);

    if (result.response) {
      await this.sendMessage(phone + '@c.us', result.response);
    }

    // Special actions after commands
    if (result.success && result.action) {
      switch (result.action) {
        case 'ticket_created':
          // Start ticket monitoring
          notifications.startTicketMonitoring(result.ticketId, phone);
          
          // Notify administrators
          await this.notifyAdminsNewTicket(result.ticketId, phone);
          break;

        case 'ticket_accepted':
          // Stop timeout monitoring
          notifications.stopTicketMonitoring(result.ticketId);
          
          // Send old ticket messages to agent
          await this.forwardPreviousMessagesToAgent(result.ticketId, phone);
          
          // Notify client
          await notifications.notifyClientAgentStarted(
            result.clientPhone,
            result.agentName,
            result.ticketId,
            result.clientLanguage
          );
          break;

        case 'ticket_closed':
          // Stop all monitoring
          notifications.stopTicketMonitoring(result.ticketId);
          notifications.stopAgentResponseMonitoring(result.ticketId);
          
          // Notify client
          await notifications.notifyClientTicketClosed(
            result.clientPhone,
            result.ticketId,
            result.clientLanguage
          );
          break;

        case 'transfer_accepted':
          // Send ticket history to new agent
          if (result.messages && result.messages.length > 0) {
            let historyMsg = `📋 *Ticket #${result.transfer.ticketId} - Conversation History*\n\n`;
            historyMsg += `👤 Client: ${result.transfer.clientName}\n`;
            historyMsg += `📞 Phone: ${result.transfer.clientPhone}\n`;
            historyMsg += `👥 From: ${result.transfer.fromAgent}\n`;
            historyMsg += `📝 Reason: ${result.transfer.reason}\n\n`;
            historyMsg += `━━━━━━━━━━━━━━━━━━━━━━\n\n`;
            
            for (const msg of result.messages) {
              const sender = msg.sender_type === 'client' ? '👤 Client' : '👨‍💼 Agent';
              const timestamp = new Date(msg.created_at).toLocaleString('en-US');
              historyMsg += `${sender} [${timestamp}]:\n${msg.content}\n\n`;
            }
            
            await this.sendMessage(phone + '@c.us', historyMsg);
          }
          
          // Notify original agent that transfer was accepted
          await notifications.notifyAgentTransferAccepted(
            result.transfer.fromAgent,
            result.newAgentName,
            result.transfer.clientName
          );
          
          // Notify client about new agent (if requested)
          if (result.transfer.notifyClient) {
            const ticket = database.getTicket(result.transfer.ticketId);
            await notifications.notifyClientNewAgent(
              result.transfer.clientPhone,
              result.newAgentName,
              ticket.language
            );
          }
          break;

        case 'notification_sent':
          // Send notification context to client
          const contextMsgEN = `📬 *Notification from Our Team*

Hello! This is a notification about a previous conversation you had with our support.

⚠️ *IMPORTANT:* This is NOT an active ticket conversation.

📋 If you need support now, please use:
/open ticket

❓ To see all available commands:
/help

━━━━━━━━━━━━━━━━━━━━━━`;

          const contextMsg = await translator.translateMessage(contextMsgEN, result.clientLanguage);
          await this.sendMessage(result.clientPhone + '@c.us', contextMsg);

          // Send notification messages
          for (const msg of result.messages) {
            if (msg.type === 'text') {
              // Translate agent message to client language
              const translatedContent = await translator.translateToClient(msg.content, result.clientLanguage);
              await this.sendMessage(result.clientPhone + '@c.us', translatedContent);
            } else {
              // Media - just inform it was included
              await this.sendMessage(result.clientPhone + '@c.us', `📎 [${msg.type}] ${msg.content}`);
            }
          }

          console.log(`✅ Notification delivered to ${result.clientPhone}`);
          break;
      }
    }
  }

  /**
   * Processa mensagem de cliente
   */
  async handleClientMessage(phone, message) {
    const content = message.body;
    const messageType = message.type;
    
    console.log(`🔍 handleClientMessage: phone=${phone}, type=${messageType}`);

    // Verifica se cliente precisa confirmar idioma
    const client = database.getClient(phone);
    const ticket = database.getOpenTicket(phone);
    
    // IMPORTANTE: Se já tem ticket aberto, não bloqueia mídia por idioma
    if (client && !client.language_confirmed && !ticket) {
      // Só bloqueia se NÃO tiver ticket aberto
      if (messageType !== 'chat') {
        console.log(`⚠️ Cliente ${phone} enviou ${messageType} mas não confirmou idioma e não tem ticket`);
        return;
      }
      // Cliente precisa escolher idioma (1-12 para popular languages, ou código ISO)
      const selection = content.trim().toLowerCase();
      if (/^([1-9]|1[0-2]|[a-z]{2,3})$/.test(selection)) {
        console.log(`✅ Cliente ${phone} escolheu idioma: ${selection}`);
        await this.handleLanguageSelection(phone, selection);
        return;
      }
    }

    // Verifica se cliente está em fluxo de criação de ticket
    const clientState = commands.getClientState(phone);
    if (clientState && clientState.flow === 'ticket_creation') {
      // Processa fluxo de ticket
      if (messageType === 'chat') {
        const result = await commands.processTicketFlow(phone, content);
        if (result && result.response) {
          await this.sendMessage(phone + '@c.us', result.response);
        }
        return;
      } else {
        // Adiciona multimídia ao fluxo
        await this.handleMediaInTicketFlow(phone, message);
        return;
      }
    }

    // Verifica se cliente está em fluxo de fechamento de ticket
    if (clientState && clientState.flow === 'close_ticket') {
      if (messageType === 'chat') {
        const result = await commands.processCloseTicketFlow(phone, content);
        if (result) {
          await this.sendMessage(phone + '@c.us', result.response);
          
          // Se fechou o ticket, processa a ação
          if (result.action === 'ticket_closed_by_client') {
            await notifications.notifyAgentsClientClosedTicket(
              result.ticketId,
              result.clientPhone,
              result.clientName,
              result.agentPhone,
              result.reason
            );
            await notifications.notifyClientTicketClosed(
              result.clientPhone,
              result.ticketId,
              result.language,
              result.reason
            );
          }
        }
        return;
      }
      return;
    }

    // Verifica se cliente está em fluxo de edição de perfil
    if (clientState && clientState.flow === 'profile_edit') {
      if (messageType === 'chat') {
        const result = await commands.processProfileEditFlow(phone, content);
        if (result && result.response) {
          await this.sendMessage(phone + '@c.us', result.response);
        }
        return;
      }
      return;
    }

    // ticket já foi buscado acima, reutiliza

    if (!ticket) {
      // Cliente não tem ticket
      // Se enviou mídia sem ter ticket, ignora
      if (messageType !== 'chat') {
        console.log(`⚠️ Cliente ${phone} enviou ${messageType} mas não tem ticket aberto`);
        return;
      }
      
      // Cliente sem ticket - envia boas-vindas (já verifica se é antigo ou novo)
      await this.sendWelcomeMessage(phone);
      return;
    }

    // Verifica horário de atendimento APENAS se não tiver agente atendendo
    if (!businessHours.isBusinessHours() && !ticket.agent_name) {
      // Fora do horário E sem agente - envia mensagem automática
      const language = client ? client.language : 'en';
      const outOfHoursMsgEN = `🕐 *Outside Business Hours*

We received your message! Our team is currently outside business hours.

⏰ Hours: ${config.businessHours.start} - ${config.businessHours.end}
📅 ${businessHours.getBusinessDaysText()}

Your ticket has been registered and we will respond as soon as possible.
Thank you for your understanding! 🙏`;
      
      const outOfHoursMsg = await translator.translateMessage(outOfHoursMsgEN, language);
      await this.sendMessage(phone + '@c.us', outOfHoursMsg);
      
      // Registra mensagem no ticket mesmo fora do horário
      await this.saveClientMessageToTicket(phone, message, ticket, language);
      return;
    }
    
    // Se tem agente atendendo, continua normalmente mesmo fora do horário
    console.log(ticket.agent_name ? `✅ Ticket #${ticket.id} em atendimento - continua mesmo fora do horário` : `✅ Dentro do horário de atendimento`);

    // Salva mensagem no ticket
    const language = client ? client.language : 'pt';
    
    if (message.type !== 'chat') {
      console.log(`💾 Salvando ${message.type} do cliente ${phone} no ticket #${ticket.id}`);
    }
    
    await this.saveClientMessageToTicket(phone, message, ticket, language);

    // Verifica se ticket foi pulado (status = 'waiting')
    if (ticket.status === 'waiting') {
      // Ticket foi pulado - notifica que o cliente respondeu
      const notificationEN = `🔔 *Skipped Client Responded!*

📋 Ticket: #${ticket.id}
👤 Client: ${ticket.client_name}
📞 Phone: ${phone}

💬 New message received.

💡 Use: /atender or /attend ${phone} to resume service.`;

      const notification = await translator.translateSystemMessage(notificationEN);
      
      // Notify all active agents
      const agents = this.getActiveAgents();
      for (const agentNumber of agents) {
        try {
          await this.sendMessage(agentNumber + '@c.us', notification);
        } catch (error) {
          console.error(`❌ Error notifying agent ${agentNumber}:`, error.message);
        }
      }
      
      // Altera status de volta para 'open'
      database.updateTicket(ticket.id, {
        status: 'open'
      });
      
      return;
    }

    // Se ticket tem agente, encaminha mensagem traduzida
    if (ticket.agent_name && ticket.agent_number) {
      await this.forwardClientMessageToAgent(phone, message, ticket, language);
    } else {
      // Notifica admins sobre mensagem
      if (message.type === 'chat') {
        // Mensagem de texto
        const content = message.body;
        const translatedContent = await translator.translateToTeam(content, language);
        
        const notificationMsgEN = `💬 *Nova mensagem no Ticket #${ticket.id}*
👤 ${ticket.client_name} (${phone})
🌍 Idioma: ${translator.getLanguageName(language)}

${translatedContent}${content !== translatedContent ? `\n\n_Original:_ ${content}` : ''}`;
        
        const notificationMsg = await translator.translateSystemMessage(notificationMsgEN);
        // Send to all active agents
        const agents = this.getActiveAgents();
        if (agents.length > 0) {
          await this.sendMessage(agents[0] + '@c.us', notificationMsg);
        }
      } else {
        // Mídia - notifica admins
        const notificationMsgEN = `📎 *New ${message.type} in Ticket #${ticket.id}*
👤 ${ticket.client_name} (${phone})
🌍 Language: ${translator.getLanguageName(language)}

💡 Use: /atender or /attend ${phone} to view media`;

        const notificationMsg = await translator.translateSystemMessage(notificationMsgEN);
        const agents = this.getActiveAgents();
        for (const agentNumber of agents) {
          try {
            await this.sendMessage(agentNumber + '@c.us', notificationMsg);
          } catch (error) {
            console.error(`❌ Error notifying agent ${agentNumber}:`, error.message);
          }
        }
      }
      
      // Confirma recebimento para cliente
      const confirmMsg = await translator.translateToClient(
        '✅ Message received. Our team has been notified and will contact you soon.',
        language
      );
      await this.sendMessage(phone + '@c.us', confirmMsg);
    }
  }

  /**
   * Processa mensagem de agente
   */
  async handleAgentMessage(agentPhone, message) {
    console.log(`🔍 handleAgentMessage: phone=${agentPhone}, type=${message.type}`);
    
    // Encontra ticket que o agente está atendendo
    const tickets = database.listOpenTickets();
    const ticket = tickets.find(t => t.agent_number === agentPhone);

    if (!ticket) {
      console.log(`⚠️ Agente ${agentPhone} não está atendendo nenhum ticket`);
      await this.sendMessage(
        agentPhone + '@c.us',
        '⚠️ You are not attending any ticket. Use /listar tickets to see available tickets.'
      );
      return;
    }

    console.log(`✅ Agente ${agentPhone} atendendo ticket #${ticket.id} do cliente ${ticket.phone}`);

    // Salva mensagem do agente
    await this.saveAgentMessageToTicket(agentPhone, message, ticket);

    // Encaminha mensagem traduzida para o cliente
    await this.forwardAgentMessageToClient(agentPhone, message, ticket);

    // Reinicia timer de monitoramento de resposta
    notifications.stopAgentResponseMonitoring(ticket.id);
  }

  /**
   * Send welcome message
   */
  async sendWelcomeMessage(phone) {
    // Verifica se cliente já existe
    const existingClient = database.getClient(phone);
    
    // Cliente que já confirmou idioma - usa idioma escolhido
    if (existingClient && existingClient.language_confirmed === 1) {
      const language = existingClient.language;
      
      // Se tem nome, mensagem personalizada
      if (existingClient.name) {
        const messageEN = `👋 *Welcome back, ${existingClient.name}!*

Great to see you again! I am *${config.bot.name}*, your virtual assistant at *${config.bot.company}*.

━━━━━━━━━━━━━━━━━━━━━━━━━━━
🎯 *Quick Actions*
━━━━━━━━━━━━━━━━━━━━━━━━━━━

📋 *Need help?*
/open ticket - Open a support ticket

👤 *Your Account*
/profile - View your information
/edit profile - Update your data

🌐 *Settings*
/change language - Change language
/help - See all commands

━━━━━━━━━━━━━━━━━━━━━━━━━━━

💡 All messages are automatically translated to your language.

How can I assist you today? 😊`;
        
        const welcomeBack = await translator.translateMessage(messageEN, language);
        await this.sendMessage(phone + '@c.us', welcomeBack);
        return;
      }
      
      // No name, but language confirmed
      const messageEN = `👋 *Welcome back to ${config.bot.company}!*

I am *${config.bot.name}*, your intelligent virtual assistant.

━━━━━━━━━━━━━━━━━━━━━━━━━━━
🎯 *How Can I Help?*
━━━━━━━━━━━━━━━━━━━━━━━━━━━

📋 *Need support?*
/open ticket

👤 *Update your profile?*
/edit profile

❓ *Need help?*
/help

━━━━━━━━━━━━━━━━━━━━━━━━━━━

💡 All messages are automatically translated.`;
      
      const welcomeBack = await translator.translateMessage(messageEN, language);
      await this.sendMessage(phone + '@c.us', welcomeBack);
      return;
    }
    
    // Novo cliente OU idioma não confirmado
    const detectedInfo = translator.identifyLanguageByPhone(phone);
    const detectedLanguage = detectedInfo.language;
    const countryName = detectedInfo.countryName;
    
    // Salva cliente com idioma sugerido (não confirmado)
    database.saveClient(phone, {
      language: detectedLanguage,
      language_confirmed: 0
    });

    // Improved bilingual welcome message
    const welcomemessageEN = `👋 *Welcome to ${config.bot.company}!*

Hello! I am *${config.bot.name}*, your intelligent virtual assistant, here to help you 24/7.

━━━━━━━━━━━━━━━━━━━━━━━━━━━
🌍 *Language Detected*
━━━━━━━━━━━━━━━━━━━━━━━━━━━

📍 Country: *${countryName}*
🗣️ Language: *${translator.getLanguageName(detectedLanguage)}*

Your messages will be automatically translated to this language.

━━━━━━━━━━━━━━━━━━━━━━━━━━━
🎯 *Quick Start*
━━━━━━━━━━━━━━━━━━━━━━━━━━━

📋 *Need support?*
Type: /open ticket

🌐 *Change language?*
Type: /change language [code]
Example: /change language en

❓ *See all commands?*
Type: /help

━━━━━━━━━━━━━━━━━━━━━━━━━━━

We're excited to assist you! 🚀`;

    const welcomeMessageEN = `

━━━━━━━━━━━━━━━━━━━━━━━━━━━
📢 *NOTICE - English Fallback*
━━━━━━━━━━━━━━━━━━━━━━━━━━━

You are seeing this message in English to ensure you understand how our system works.

✅ *Automatic Detection:*
We detected your language based on your phone number (${detectedInfo.dialCode}).

🔄 *Wrong language detected?*
Change anytime with: /change language [code]

💡 *Available languages:*
pt (Portuguese), en (English), es (Spanish), fr (French), de (German), it (Italian), zh (Chinese), ja (Japanese), ko (Korean), hi (Hindi), and many more!

Type /help to see all available commands.`;
    
    // Traduz para o idioma detectado
    const translatedWelcome = await translator.translateMessage(welcomemessageEN, detectedLanguage);
    
    // Combina mensagem traduzida + aviso em inglês
    const finalMessage = translatedWelcome + welcomeMessageEN;
    
    await this.sendMessage(phone + '@c.us', finalMessage);

    // Language selection is now handled via /change language command
    // No automatic language selection prompt on first contact
  }

  /**
   * Gera mensagem de boas-vindas
   */
  async getWelcomeMessage(language) {
    const messages = {
      pt: `👋 *Olá! Eu sou ${config.bot.name}, bot da ${config.bot.company}.*

Vou conectá-lo com nossos agentes para o melhor atendimento possível!

🌍 *Hello! I'm ${config.bot.name}, bot from ${config.bot.company}.*

I'll connect you with our agents for the best service possible!`,

      en: `👋 *Hello! I'm ${config.bot.name}, bot from ${config.bot.company}.*

I'll connect you with our agents for the best service possible!

🌍 *Olá! Eu sou ${config.bot.name}, bot da ${config.bot.company}.*

Vou conectá-lo com nossos agentes para o melhor atendimento possível!`,

      es: `👋 *¡Hola! Soy ${config.bot.name}, bot de ${config.bot.company}.*

¡Te conectaré con nuestros agentes para el mejor servicio posible!

🌍 *Hello! I'm ${config.bot.name}, bot from ${config.bot.company}.*

I'll connect you with our agents for the best service possible!`
    };

    return messages[language] || messages.pt;
  }

  /**
   * Salva mensagem do cliente no ticket
   */
  async saveClientMessageToTicket(phone, message, ticket, language) {
    let content = message.body;
    let mediaUrl = null;
    let messageType = 'text';

    // Se for multimídia, processa e salva arquivo
    if (message.type !== 'chat') {
      console.log(`💾 SALVANDO MÍDIA DO CLIENTE: tipo=${message.type}, phone=${phone}, ticket=${ticket.id}`);
      messageType = message.type;
      
      // Salva arquivo físico
      const client = database.getClient(phone);
      const clientName = client?.name || ticket.client_name || 'cliente';
      console.log(`📝 Nome do cliente: ${clientName}`);
      
      mediaUrl = await mediaHandler.saveMedia(this.client, message, phone, clientName, false);
      console.log(`✅ Arquivo salvo em: ${mediaUrl}`);
      
      content = message.caption || `[${messageType}]`;
    }

    // Traduz para português (idioma da equipe)
    const translatedContent = await translator.translateToTeam(content, language);

    // Salva no banco
    database.addMessage(
      ticket.id,
      phone,
      'client',
      content,
      messageType,
      translatedContent,
      mediaUrl,
      language,
      'en'
    );
  }

  /**
   * Salva mensagem do agente no ticket
   */
  async saveAgentMessageToTicket(agentPhone, message, ticket) {
    let content = message.body;
    let mediaUrl = null;
    let messageType = 'text';

    if (message.type !== 'chat') {
      console.log(`💾 SALVANDO MÍDIA DO AGENTE: tipo=${message.type}, phone=${agentPhone}, ticket=${ticket.id}`);
      messageType = message.type;
      
      // Salva arquivo físico (pasta admin)
      mediaUrl = await mediaHandler.saveMedia(this.client, message, agentPhone, 'admin', true);
      console.log(`✅ Arquivo do agente salvo em: ${mediaUrl}`);
      
      content = message.caption || `[${messageType}]`;
    }

    // Traduz para idioma do cliente
    const translatedContent = await translator.translateToClient(content, ticket.language);

    // Salva no banco
    database.addMessage(
      ticket.id,
      ticket.phone,
      'agent',
      content,
      messageType,
      translatedContent,
      mediaUrl,
      'en',
      ticket.language
    );
  }

  /**
   * Encaminha mensagem do cliente para o agente
   */
  async forwardClientMessageToAgent(clientPhone, message, ticket, language) {
    // Se for multimídia
    if (message.type !== 'chat') {
      try {
        console.log(`📤 Cliente ${clientPhone} enviou ${message.type} para agente ${ticket.agent_number}`);
        
        // Envia cabeçalho
        const header = `📨 *Mídia de ${ticket.client_name}* (Ticket #${ticket.id})
🌍 Idioma: ${translator.getLanguageName(language)}
📎 Tipo: ${message.type}`;
        
        await this.sendMessage(ticket.agent_number + '@c.us', header);
        
        // Baixa e envia mídia
        const buffer = await this.client.decryptFile(message);
        let mimetype = message.mimetype || 'application/octet-stream';
        
        // Limpa mimetype (remove codecs e espaços)
        mimetype = mimetype.split(';')[0].trim();
        
        const filename = `media.${mediaHandler.getExtension(mimetype, message.type)}`;
        let caption = message.caption || '';
        
        // Translate caption if exists
        if (caption) {
          const teamLanguage = config.translation.defaultLanguage;
          if (language !== teamLanguage) {
            try {
              const translatedCaption = await translator.translateToTeam(caption, language);
              // Add both translated and original caption
              if (translatedCaption !== caption) {
                caption = `${translatedCaption}\n\n_Original (${translator.getLanguageName(language)}):_\n${caption}`;
              }
            } catch (error) {
              console.error('⚠️ Error translating media caption:', error.message);
            }
          }
        }
        
        // Converte buffer para base64
        const base64 = `data:${mimetype};base64,${buffer.toString('base64')}`;
        
        console.log(`📋 Tipo: ${message.type}, Mimetype: ${mimetype}, Filename: ${filename}`);
        
        // Envia usando sendFile do WPP Connect
        await this.client.sendFile(
          ticket.agent_number + '@c.us',
          base64,
          filename,
          caption
        );
        
        console.log(`✅ Mídia do cliente encaminhada para agente`);
      } catch (error) {
        console.error('❌ Erro ao encaminhar mídia para agente:', error);
        console.error('Stack trace:', error.stack);
      }
    } else {
      // Mensagem de texto
      const content = message.body;
      const translatedContent = await translator.translateToTeam(content, language);

      let forwardMessage = `📨 *Mensagem de ${ticket.client_name}* (Ticket #${ticket.id})
🌍 Idioma: ${translator.getLanguageName(language)}

💬 ${translatedContent}`;

      if (content !== translatedContent) {
        forwardMessage += `\n\n_Original:_ ${content}`;
      }

      await this.sendMessage(ticket.agent_number + '@c.us', forwardMessage);
    }

    // Inicia monitoramento de resposta do agente
    notifications.startAgentResponseMonitoring(ticket.id, ticket.agent_number);
  }

  /**
   * Encaminha mensagem do agente para o cliente
   */
  async forwardAgentMessageToClient(agentPhone, message, ticket) {
    // Se for multimídia, envia arquivo
    if (message.type !== 'chat') {
      try {
        console.log(`📤 Tentando enviar ${message.type} para ${ticket.phone}`);
        
        // Baixa o arquivo
        const buffer = await this.client.decryptFile(message);
        let mimetype = message.mimetype || 'application/octet-stream';
        
        // Limpa mimetype (remove codecs e espaços)
        mimetype = mimetype.split(';')[0].trim();
        
        const filename = `media.${mediaHandler.getExtension(mimetype, message.type)}`;
        let caption = message.caption || '';
        
        // Translate caption if exists
        if (caption) {
          try {
            const translatedCaption = await translator.translateToClient(caption, ticket.language);
            
            // If different from original, add both translated and original
            if (translatedCaption !== caption && ticket.language !== config.translation.defaultLanguage) {
              caption = `${translatedCaption}\n\n⚠️ _Automatically translated message_`;
            } else {
              caption = translatedCaption;
            }
          } catch (error) {
            console.error('⚠️ Error translating media caption to client:', error.message);
          }
        }
        
        // Converte buffer para base64
        const base64 = `data:${mimetype};base64,${buffer.toString('base64')}`;
        
        console.log(`📋 Tipo: ${message.type}, Mimetype: ${mimetype}, Filename: ${filename}`);
        
        // Envia usando sendFile do WPP Connect
        await this.client.sendFile(
          ticket.phone + '@c.us',
          base64,
          filename,
          caption
        );
        
        console.log(`✅ Mídia encaminhada para ${ticket.phone}`);
      } catch (error) {
        console.error('❌ Erro ao encaminhar mídia:', error);
        console.error('Stack trace:', error.stack);
        
        // Tenta enviar mensagem de erro ao cliente
        try {
          const errorMsg = await translator.translateToClient(
            'Desculpe, houve um erro ao enviar a mídia. O agente foi notificado.',
            ticket.language
          );
          await this.sendMessage(ticket.phone + '@c.us', errorMsg);
        } catch (e) {
          console.error('❌ Erro ao enviar mensagem de erro:', e.message);
        }
      }
      return;
    }

    // Se for texto, traduz e envia
    const content = message.body || message.caption || '';
    if (content) {
      console.log(`🔄 Translating agent message to client:`);
      console.log(`   Original (${config.translation.defaultLanguage}): ${content}`);
      console.log(`   Target language: ${ticket.language}`);
      
      const translatedContent = await translator.translateToClient(content, ticket.language);
      
      console.log(`   Translated: ${translatedContent}`);
      
      // Adiciona cabeçalho e aviso de tradução
      const agentName = ticket.agent_name || 'Agent';
      
      const headers = {
        pt: `👤 *Resposta do ${agentName}:*`,
        en: `👤 *Response from ${agentName}:*`,
        es: `👤 *Respuesta de ${agentName}:*`,
        fr: `👤 *Réponse de ${agentName}:*`,
        de: `👤 *Antwort von ${agentName}:*`,
        it: `👤 *Risposta da ${agentName}:*`,
        zh: `👤 *${agentName}的回复:*`,
        ja: `👤 *${agentName}からの返信:*`,
        ko: `👤 *${agentName}의 응답:*`,
        hi: `👤 *${agentName} का जवाब:*`
      };
      
      const warnings = {
        pt: `\n\n⚠️ _Mensagem traduzida automaticamente. Possíveis erros podem ocorrer._`,
        en: `\n\n⚠️ _Automatically translated message. Possible errors may occur._`,
        es: `\n\n⚠️ _Mensaje traducido automáticamente. Pueden ocurrir posibles errores._`,
        fr: `\n\n⚠️ _Message traduit automatiquement. Des erreurs possibles peuvent survenir._`,
        de: `\n\n⚠️ _Automatisch übersetzte Nachricht. Mögliche Fehler können auftreten._`,
        it: `\n\n⚠️ _Messaggio tradotto automaticamente. Possibili errori possono verificarsi._`,
        zh: `\n\n⚠️ _自动翻译的消息。可能会出现错误._`,
        ja: `\n\n⚠️ _自動翻訳されたメッセージ。エラーが発生する可能性があります._`,
        ko: `\n\n⚠️ _자동 번역된 메시지입니다. 오류가 발생할 수 있습니다._`,
        hi: `\n\n⚠️ _स्वचालित रूप से अनुवादित संदेश। त्रुटियाँ हो सकती हैं._`
      };
      
      const header = headers[ticket.language] || headers.en;
      const warning = warnings[ticket.language] || warnings.en;
      
      const finalMessage = `${header}\n\n${translatedContent}${content !== translatedContent ? warning : ''}`;
      
      console.log(`✅ Sending to client ${ticket.phone}: ${finalMessage.substring(0, 100)}...`);
      
      await this.sendMessage(ticket.phone + '@c.us', finalMessage);
    }
  }

  /**
   * Processa seleção de idioma
   */
  async handleLanguageSelection(phone, selection) {
    const language = translator.mapSelectionToLanguage(selection);
    
    if (language) {
      database.setClientLanguage(phone, language);

      const languageName = translator.getLanguageName(language);
      const confirmMessageEN = `✅ Language set to ${languageName}!

To open a support ticket, use the command:
/open ticket

💡 Your messages will be automatically translated to our team.`;
      
      const confirmMessage = await translator.translateToClient(confirmMessageEN, language);

      await this.sendMessage(phone + '@c.us', confirmMessage);
    } else {
      // Invalid selection
      const errorMsgEN = `❌ Invalid language selection.

Please type a number (1-12) or a valid ISO language code (e.g., en, pt, es).

Type /change language to see the list.`;
      
      await this.sendMessage(phone + '@c.us', errorMsgEN);
    }
  }

  /**
   * Adiciona multimídia ao fluxo de ticket
   */
  async handleMediaInTicketFlow(phone, message) {
    const state = commands.getClientState(phone);
    const client = database.getClient(phone);
    const language = client?.language || 'pt';
    
    if (state && state.step === 'description') {
      // Valida tamanho da mídia
      if (message.size) {
        const sizeMB = message.size / (1024 * 1024);
        const maxSizeMB = config.media.maxSizeMB;
        
        if (sizeMB > maxSizeMB) {
          const errorMsgEN = `⚠️ *Arquivo muito grande*

Tamanho do arquivo: ${sizeMB.toFixed(2)} MB
Tamanho máximo permitido: ${maxSizeMB} MB

Por favor, envie um arquivo menor ou comprima-o.`;
          
          const translatedError = await translator.translateMessage(errorMsgEN, language);
          await this.sendMessage(phone + '@c.us', translatedError);
          return;
        }
      }
      
      if (!state.data.messages) {
        state.data.messages = [];
      }

      // Store media with base64 data for later forwarding
      const mediaInfo = {
        type: message.type,
        content: message.caption || `[${message.type}]`,
        timestamp: new Date().toISOString()
      };

      // Save base64 media data if available
      if (message.body && message.body.startsWith('data:')) {
        mediaInfo.media_data = message.body; // Base64 data
        mediaInfo.mimetype = message.mimetype;
      } else if (message.mediaData) {
        // Try to get media data
        try {
          const mediaData = await this.client.decryptFile(message);
          mediaInfo.media_data = `data:${message.mimetype};base64,${mediaData.toString('base64')}`;
          mediaInfo.mimetype = message.mimetype;
        } catch (error) {
          console.error('⚠️ Error decrypting media:', error.message);
          mediaInfo.media_url = message.mediaData?.filename || 'media';
        }
      }

      state.data.messages.push(mediaInfo);
      commands.updateClientState(phone, state);

      const successMsgEN = `✅ ${message.type} added. Continue sending or use /send when finished.`;
      const translatedSuccess = await translator.translateMessage(successMsgEN, language);
      
      await this.sendMessage(phone + '@c.us', translatedSuccess);
    }
  }

  /**
   * Notifica administradores sobre novo ticket
   */
  async notifyAdminsNewTicket(ticketId, clientPhone) {
    const ticket = database.getTicket(ticketId);
    const messages = database.getTicketMessages(ticketId);
    
    // Pega primeira mensagem como preview
    let preview = '';
    if (messages && messages.length > 0) {
      const firstMsg = messages[0];
      preview = firstMsg.content || firstMsg.message_type;
      if (preview.length > 100) {
        preview = preview.substring(0, 100) + '...';
      }
    }
    
    let messageEN = `🎫 *New Ticket Opened!*

📋 #${ticketId}
👤 Client: ${ticket.client_name}
📞 Phone: ${clientPhone}
🌍 Language: ${translator.getLanguageName(ticket.language)}`;

    if (preview) {
      messageEN += `\n📝 Preview: ${preview}`;
    }

    messageEN += `\n\n💡 Use: /atender or /attend ${clientPhone}`;

    const message = await translator.translateSystemMessage(messageEN);
    const agents = this.getActiveAgents();
    for (const agentNumber of agents) {
      try {
        await this.sendMessage(agentNumber + '@c.us', message);
      } catch (error) {
        console.error(`❌ Error notifying agent ${agentNumber}:`, error.message);
      }
    }
  }

  /**
   * Encaminha mensagens anteriores do ticket para o agente
   */
  async forwardPreviousMessagesToAgent(ticketId, agentPhone) {
    try {
      const messages = database.getTicketMessages(ticketId);
      const ticket = database.getTicket(ticketId);
      
      if (!messages || messages.length === 0) {
        return;
      }

      // Send translated header
      const headerEN = `📋 *Ticket History #${ticketId}*\n👤 Client: ${ticket.client_name}\n🌍 Language: ${translator.getLanguageName(ticket.language)}\n\n━━━━━━━━━━━━━━━━━━━━\n`;
      const header = await translator.translateSystemMessage(headerEN);
      await this.sendMessage(agentPhone + '@c.us', header);

      // Send each message
      for (const msg of messages) {
        if (msg.sender_type === 'client') {
          const teamLanguage = config.translation.defaultLanguage;
          
          // Check if message has media
          if (msg.message_type !== 'text' && msg.media_url) {
            // Send media with translated caption
            const clientLabelEN = '📎 *Client sent media:*';
            const clientLabel = await translator.translateSystemMessage(clientLabelEN);
            let caption = `${clientLabel}\n`;
            caption += `🗂️ Type: ${msg.message_type}\n`;
            
            // If message has text content (caption), translate it
            if (msg.content) {
              const originalContent = msg.original_content || msg.content;
              let translatedCaption = msg.content;
              
              if (ticket.language !== teamLanguage) {
                try {
                  translatedCaption = await translator.translateToTeam(originalContent, ticket.language);
                } catch (error) {
                  console.error('⚠️ Error translating caption:', error.message);
                  translatedCaption = originalContent;
                }
              }
              
              caption += `💬 ${translatedCaption}`;
              
              // Show original if translated
              if (translatedCaption !== originalContent && ticket.language !== teamLanguage) {
                const originalLabelEN = `\n\n_Original (${translator.getLanguageName(ticket.language)}):_`;
                const originalLabel = await translator.translateSystemMessage(originalLabelEN);
                caption += `${originalLabel}\n${originalContent}`;
              }
            }
            
            caption += `\n\n⏰ ${new Date(msg.created_at).toLocaleString('en-US', { timeZone: 'Africa/Maputo' })}`;
            caption += `\n━━━━━━━━━━━━━━━━━━━━`;
            
            // Send media message
            try {
              // Check if we have base64 data or URL
              const mediaSource = msg.media_url;
              
              if (mediaSource && mediaSource.startsWith('data:')) {
                // It's base64 data
                await this.client.sendFile(
                  agentPhone + '@c.us',
                  mediaSource,
                  `media_${msg.message_type}`,
                  caption
                );
              } else if (mediaSource) {
                // It's a file path or URL
                await this.client.sendFile(
                  agentPhone + '@c.us',
                  mediaSource,
                  '',
                  caption
                );
              } else {
                // No media available, send caption only
                console.warn('⚠️ Media not available for message:', msg.id);
                await this.sendMessage(agentPhone + '@c.us', caption + `\n\n⚠️ Media not available`);
              }
            } catch (error) {
              console.error('❌ Error sending media to agent:', error.message);
              // Send error notification
              await this.sendMessage(agentPhone + '@c.us', caption + `\n\n⚠️ Error loading media`);
            }
          } else {
            // Text message - ALWAYS translate to agent's language (configured in .env)
            const originalContent = msg.original_content || msg.content;
            let translatedToTeam = msg.content;
            
            // If ticket language is different from team language, translate
            if (ticket.language !== teamLanguage) {
              try {
                translatedToTeam = await translator.translateToTeam(originalContent, ticket.language);
              } catch (error) {
                console.error('⚠️ Error translating history:', error.message);
                translatedToTeam = originalContent;
              }
            }
            
            // Format message: translation in team language + original in client language
            const clientLabelEN = '💬 *Client:*';
            const clientLabel = await translator.translateSystemMessage(clientLabelEN);
            let msgText = `${clientLabel}\n${translatedToTeam}`;
            
            // If translated, show original
            if (translatedToTeam !== originalContent && ticket.language !== teamLanguage) {
              const originalLabelEN = `_Original (${translator.getLanguageName(ticket.language)}):_`;
              const originalLabel = await translator.translateSystemMessage(originalLabelEN);
              msgText += `\n\n${originalLabel}\n${originalContent}`;
            }
            
            msgText += `\n\n⏰ ${new Date(msg.created_at).toLocaleString('en-US', { timeZone: 'Africa/Maputo' })}`;
            msgText += `\n━━━━━━━━━━━━━━━━━━━━`;
            
            await this.sendMessage(agentPhone + '@c.us', msgText);
          }
          
          // Small delay between messages
          await new Promise(resolve => setTimeout(resolve, 800));
        }
      }

      // Send translated footer
      const footerEN = '\n✅ *End of history*\nYou can start replying now.';
      const footer = await translator.translateSystemMessage(footerEN);
      await this.sendMessage(agentPhone + '@c.us', footer);
      
    } catch (error) {
      console.error('❌ Erro ao enviar histórico:', error.message);
    }
  }

  /**
   * Envia mensagem via WPPConnect
   */
  async sendMessage(to, text) {
    if (!this.client) {
      console.error('❌ Cliente WPPConnect não inicializado');
      return;
    }

    try {
      await this.client.sendText(to, text);
      console.log(`✅ Mensagem enviada para ${to}`);
    } catch (error) {
      console.error(`❌ Erro ao enviar mensagem para ${to}:`, error.message);
    }
  }
}

module.exports = new MessageHandler();
